' NOMIS, a Simon Clone game for CMM2
' Rev 1.0.0 William M Leue 12-August-2023
' Rev 2.0.0 works with either a mouse or keyboard.
'  Adds an option to rotate the buttons by 45 degrees so that they point
'  in the cardinal directions.
' Rev 2.5.0 adds number of buttons to options; revises tones; various small changes.

option default integer
option base 1
option angle degrees

' Constants
const VERSION$ = "V 2.5.0"

' USE THIS CONSTANT TO CONFIGURE FOR MOUSE OR KEYBOARD CONTROLS
const USE_MOUSE      = 1   ' set to 0 for keyboard only

' general game params
const MIN_BUTTONS = 4
const MAX_BUTTONS = 8
const MAX_BEEPS = 20
const CASE_COLOR = rgb(black)
const NUM_WHITE_NOTES = 22
const NUM_BLACK_NOTES = 16
const WHITE_KEY = 1
const BLACK_KEY = 0
const USER = 1
const COMPUTER = 2
const BUTTON_BRIGHT_RATIO = 0.3765
const NSHUFFLES = 5
const SEQ_DELAY = 1000

' display params
const CASE_RAD         = 250
const BUTTON_MARGIN    = 20
const BUTTON_INNER_RAD = 60
const BUTTON_OUTER_RAD = CASE_RAD - BUTTON_MARGIN
const BUTTON_HVGAP     = int((1.0*CASE_RAD)/(20.0) + 0.5)
const BUTTON_NRADPTS   = 20
const BUTTON_NVERTS    = 4+2*BUTTON_NRADPTS

' keyboard codes
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const KQ    = 81
const KW    = 87
const KA    = 65
const KS    = 83
const KI    = 73
const KO    = 79
const KK    = 75
const KL    = 76
const ESC   = 27
const F1    = 145
const F2    = 146
const F4    = 148

' tunes and flourish params
const NWNOTES   = 6
const NDNOTES   = 8
const WIN_TUNE  = 1
const LOSE_TUNE = 2
const F_LENGTH  = 100

' Give the user a small break before timing out
const UTIMEADD  = 0.30

' Controls
const HLPX  = 10
const CTBY  = 5
const CTBW  = 80
const CTBG  = 20
const CTBH  = 30
const HLPH  = CTBH
const HLPR  = CTBH\2
const OPTX  = HLPX+CTBW+CTBG
const OPTR  = CTBH\2
const PLAX  = HLPX+2*CTBW+2*CTBG
const QITX  = HLPX+3*CTBW+3*CTBG
const CTBTW = 4*CTBW + 3*CTBG
const CTBC  = rgb(0, 128, 0)
const CTBC_FLASH = rgb(green)
const CTL_HLP = 1
const CTL_OPT = 2
const CTL_PLA = 3
const CTL_QIT = 4

' Options
const OPT_NBUTTONS = 1
const OPT_NBEEPS   = 2
const OPT_SPEED    = 3
const OPT_FONLY    = 4
const OPT_TONLY    = 5
const OPT_BACKW    = 6
const OPT_STARTA   = 7
const NOPTS = 7

' Mouse I2C channel
const MCHAN = 2

' Globals
dim prebeeps(MAX_BEEPS)
dim beeps(MAX_BEEPS)
dim buttons(MAX_BUTTONS)
dim ON_colors(MAX_BUTTONS)
dim OFF_colors(MAX_BUTTONS)
dim white_notes(NUM_WHITE_NOTES)
dim black_notes(NUM_BLACK_NOTES)
dim button_tones(MAX_BUTTONS, MAX_BUTTONS)
dim bvertx(MAX_BUTTONS, BUTTON_NVERTS)
dim bverty(MAX_BUTTONS, BUTTON_NVERTS)
dim wnotes(NWNOTES)
dim dnotes(NDNOTES)
dim windices(NWNOTES) = (12, 13, 14, 16, 14, 16)
dim wlengths(NWNOTES) = (100, 100, 100, 300, 100, 300)
dim dindices(NDNOTES) = (5, 4, 5, 3, 5, 4, 3, 3)
dim dlengths(NDNOTES) = (500, 500, 500, 500, 500, 1000, 200, 600)
dim noptvals(NOPTS) = (5, 3, 3, 2, 2, 2, 2)
dim option_values(NOPTS, 5)
dim option_text$(NOPTS)
dim option_choice(NOPTS)
dim nbuttons   = 4
dim game_beeps = 0
dim game_speed = 0
dim flash_only = 0
dim tones_only = 0
dim backwards  = 0
dim nbeeps     = 0
dim beep_length = 0
dim beep_delay = 0
dim cx = mm.hres\2
dim cy = mm.vres\2
dim mouse_x    = 0
dim mouse_y    = 0
dim lclick     = 0
dim timeout    = 0
dim user_nbeeps = 0
dim start_play = 0
dim running    = 0
dim bccount    = 0
dim bcbutton   = 0
dim start_aflag = 0

' Main Program
open "debug.txt" for output as #1
if not USE_MOUSE and nbuttons <> 4 then
  cls
  print "Sorry, Only 4 buttons are supported for keyboard-only play!"
  end
end if
ReadSounds
MakeButtonPolygons nbuttons, start_aflag
MakeButtonColors
MakeOptionValues
if USE_MOUSE then InitMouse
DrawScreen
if USE_MOUSE then
  PlayLoop
else
  start_play = 0
  do
    cmd = GetKeyboardCommand()
    select case cmd
      case CTL_HLP
        ShowInstructions
      case CTL_OPT
        ShowOptions
      case CTL_PLA
        start_play = 1    
      case CTL_QIT
        Quit
    end select
    if start_play then
      start_play = 0
      InitGame 1
      PlayGame
    end if
  loop
end

' Get Keyboard commands other than game play
function GetKeyboardCommand()
  local z$, cmd, ctlb
  ctlb = 0
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case F1 : ctlb = 1
      case F2 : ctlb = 2
      case F4 : ctlb = 3
      case ESC : ctlb = 4
    end select
    if ctlb <> 0 then
      GetKeyboardCommand = ctlb
      exit do
    end if
  loop
end function

' Read the note frequency values and set up the button tones.
' (The buttons are E, C#, A, and E (one octave below), which
' is an E major chord, 2nd inversion.)
sub ReadSounds nb
  local i, j, bw, kn
  for i = 1 to NUM_WHITE_NOTES
    read white_notes(i)
  next i
  for i = 1 to NUM_BLACK_NOTES
    read black_notes(i)
  next i
  
  ' a couple of short, cheerful (?) tunes for winning and losing
  for i = 1 to NWNOTES
    wnotes(i) = white_notes(windices(i))
  next i
  for i = 1 to NDNOTES
    dnotes(i) = white_notes(dindices(i))
  next i

  ' assign tones to buttons based on number of buttons
  for i = MIN_BUTTONS to MAX_BUTTONS
    for j = 1 to i
      read bw : read kn
      if bw = WHITE_KEY then
        button_tones(i-3, j) = white_notes(kn)
      else
        button_tones(i-3, j) = black_notes(kn)
      end if
    next j
  next i
end sub

' Make arrays of option values
sub MakeOptionValues
  local i, j
print #1, "MakeOptionValues"
  for i = 1 to NOPTS
    for j = 1 to 5
      option_values(i, j) = 0
    next j
  next i
  option_text$(OPT_NBUTTONS) = "Number of Buttons:"
  option_text$(OPT_NBEEPS) = "Number of Tones in Game:"
  option_text$(OPT_SPEED) = "Game Speed:"
  option_text$(OPT_FONLY) = "Light Flashes Only:"
  option_text$(OPT_TONLY) = "Tones Only:"
  option_text$(OPT_BACKW) = "Backwards:"
  option_text$(OPT_STARTA) = "Start Angle Option:"
  option_values(OPT_NBUTTONS, 1) = 4
  option_values(OPT_NBUTTONS, 2) = 5
  option_values(OPT_NBUTTONS, 3) = 6
  option_values(OPT_NBUTTONS, 4) = 7
  option_values(OPT_NBUTTONS, 5) = 8
  option_values(OPT_NBEEPS, 1) = 8
  option_values(OPT_NBEEPS, 2) = 14
  option_values(OPT_NBEEPS, 3) = 20
  option_values(OPT_SPEED, 1) = 1
  option_values(OPT_SPEED, 2) = 2
  option_values(OPT_SPEED, 3) = 3
  option_values(OPT_FONLY, 1) = 0
  option_values(OPT_FONLY, 2) = 1
  option_values(OPT_TONLY, 1) = 0
  option_values(OPT_TONLY, 2) = 1
  option_values(OPT_BACKW, 1) = 0
  option_values(OPT_BACKW, 2) = 1
  option_values(OPT_STARTA, 1) = 0
  option_values(OPT_STARTA, 2) = 1
end sub

' Make the vertices for the polygons for the 4 buttons
' (generalized version: can handle arbitrary starting angles
' and any number of buttons.)
' Arguments:
'  nb: number of buttons (4 to 8 are the practical limits)
'  rangle: starting angle for button #1  (0 is default for normal Simon)
sub MakeButtonPolygons nb, start_aflag
  local x1, y1, x2, y2, x3, y3, x4, y4
  local i, j, blen, vx
  local float gangle, sangle, angle, ainc, gratio, rratio, bangle, rangle
  local float cratio, cangle, wangle
  blen = BUTTON_OUTER_RAD - BUTTON_INNER_RAD
  gratio = (1.0*BUTTON_INNER_RAD)/(1.0*BUTTON_HVGAP)
  rratio = PI*gratio/180.0
  gangle = asin(rratio)
  wangle = 360.0/nb
  rangle = 0.0
  if start_aflag then rangle = 0.5*wangle
  bangle = wangle
  ainc = bangle/BUTTON_NRADPTS
  cratio = PI*(1.0*BUTTON_HVGAP)/(1.0*BUTTON_INNER_RAD)/180.0
  cangle = rangle + asin(cratio)
  for i = 1 to nb
    vx = 1
    bvertx(i, vx) = cx + int(BUTTON_INNER_RAD*cos(cangle) + 0.5)
    bverty(i, vx) = cy - int(BUTTON_INNER_RAD*sin(cangle) + 0.5)
    inc vx
    bvertx(i, vx) = cx + int(BUTTON_OUTER_RAD*cos(cangle) + 0.5)
    bverty(i, vx) = cy - int(BUTTON_OUTER_RAD*sin(cangle) + 0.5)
    angle = cangle
    for j = 1 to BUTTON_NRADPTS
      inc vx
      bvertx(i, vx) = cx + int(BUTTON_OUTER_RAD*cos(angle) + 0.5)
      bverty(i, vx) = cy - int(BUTTON_OUTER_RAD*sin(angle) + 0.5)
      inc angle, ainc
    next j
    inc vx
    bvertx(i, vx) = cx + int(BUTTON_OUTER_RAD*cos(cangle+bangle) + 0.5)
    bverty(i, vx) = cy - int(BUTTON_OUTER_RAD*sin(cangle+bangle) + 0.5)
    inc vx
    bvertx(i, vx) = cx + int(BUTTON_INNER_RAD*cos(cangle+bangle) + 0.5)
    bverty(i, vx) = cy - int(BUTTON_INNER_RAD*sin(cangle+bangle) + 0.5)
    angle = cangle + bangle
    for j = 1 to BUTTON_NRADPTS
      inc vx
      bvertx(i, vx) = cx + int(BUTTON_INNER_RAD*cos(angle) + 0.5)
      bverty(i, vx) = cy - int(BUTTON_INNER_RAD*sin(angle) + 0.5)
      inc angle, -ainc
    next j
    inc cangle, wangle
  next i
end sub
    
' Make the button colors
sub MakeButtonColors
  local i, r, g, b
  local float h, s, v1, v2, hinc
  h = 0.0
  s = 1.0
  v1 = 1.0
  v2 = BUTTON_BRIGHT_RATIO
  hinc = 270.0/nbuttons
  for i = 1 to nbuttons
    HSV2RGB h, s, v1, r, g, b
    ON_Colors(i) = rgb(r, g, b)
    HSV2RGB h, s, v2, r, g, b
    OFF_Colors(i) = rgb(r, g, b)
    inc h, hinc
  next i
end sub

' Initialize the mouse and cursor if we have one
sub InitMouse
  on error  skip 1
  controller mouse open MCHAN, LeftClick
  if mm.errno <> 0 then
    cls
    print "Sorry, this version of NOMIS requires a mouse!"
    print "Set the constant USE_MOUSE to zero to play with keyboard only."
    print "See the top of the source code for this and other options."
    end
  end if
  gui cursor on 1
  settick 20, UpdateCursor, 1
end sub

' Mouse Left Click ISR
sub LeftClick
  mouse_x = mouse(X)
  mouse_y = mouse(Y)
  lclick = 1
end sub

' Update cursor to track the mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Draw the screen
sub DrawScreen
  local i
  cls rgb(gray)
  if USE_MOUSE then
    DrawControls 0
  else
    text 10, 10, "F1=Help, F2=Options, F4=Play Game, ESC = Quit",,,, rgb(black), -1
  end if
  circle cx, cy, CASE_RAD,,, CASE_COLOR, CASE_COLOR
  for i = 1 to nbuttons
    DrawButton i, 0
  next i
  text mm.hres-1, mm.vres-1, VERSION$, "RB", 7,, rgb(black), -1
end sub

' Draw the Controls
' If which > 0 then flash the corresponding control button brighter
sub DrawControls which
  local c(4) = (CTBC, CTBC, CTBC, CTBC)
  if which > 0 then c(which) = CTBC_FLASH
  if USE_MOUSE then  gui cursor hide
  rbox HLPX, CTBY, CTBW, CTBH, HLPR, rgb(black), c(1)
  text HLPX+CTBW\2, CTBY+CTBH\2, "Help", "CM",,, rgb(black), -1
  rbox OPTX, CTBY, CTBW, CTBH, OPTR, rgb(black), c(2)
  text OPTX+CTBW\2, CTBY+CTBH\2, "Options", "CM",,, rgb(black), -1
  rbox PLAX, CTBY, CTBW, CTBH, OPTR, rgb(black), c(3)
  text PLAX+CTBW\2, CTBY+CTBH\2, "Play!", "CM",,, rgb(black), -1
  rbox QITX, CTBY, CTBW, CTBH, OPTR, rgb(black), c(4)
  text QITX+CTBW\2, CTBY+CTBH\2, "Quit", "CM",,, rgb(black), -1
  if USE_MOUSE then gui cursor show
end sub

' Draw the specified button with dim colors (not active) or bright (active)
sub DrawButton which, active
  local c
  local xv(BUTTON_NVERTS), yv(BUTTON_NVERTS)
  if USE_MOUSE then gui cursor hide
  if active then
    c = ON_colors(which)
  else
    c = OFF_colors(which)
  end if
  math slice bvertx(), which,, xv()
  math slice bverty(), which,, yv()
  polygon BUTTON_NVERTS, xv(), yv(), c, c
  text cx, cy, "NOMIS", "CM", 4
  if USE_MOUSE then gui cursor show
  DrawButtonGaps
end sub

' Draw the black gaps between buttons
sub DrawButtonGaps
  local i
  local sxv(4), syv(4)
  local float angle, ainc, pangle, sangle
  local r1, r2, x1, y1, x2, y2
  ainc = 360.0/nbuttons
  sangle = 0
  if start_aflag then sangle = 0.5*ainc
  for i = 1 to nbuttons
    angle = sangle + (i-1)*ainc
    r1 = BUTTON_INNER_RAD - 4
    r2 = BUTTON_OUTER_RAD + 4
    x1 = cx + int(r1*cos(angle) + 0.5)
    y1 = cy - int(r1*sin(angle) + 0.5)
    x2 = cx + int(r2*cos(angle) + 0.5)
    y2 = cy - int(r2*sin(angle) + 0.5)
    pangle = angle + 90.0
    sxv(1) = x1 + int(BUTTON_HVGAP*cos(pangle) + 0.5)
    syv(1) = y1 - int(BUTTON_HVGAP*sin(pangle) + 0.5)
    sxv(2) = x2 + int(BUTTON_HVGAP*cos(pangle) + 0.5)
    syv(2) = y2 - int(BUTTON_HVGAP*sin(pangle) + 0.5)
    sxv(3) = x2 - int(BUTTON_HVGAP*cos(pangle) + 0.5)
    syv(3) = y2 + int(BUTTON_HVGAP*sin(pangle) + 0.5)
    sxv(4) = x1 - int(BUTTON_HVGAP*cos(pangle) + 0.5)
    syv(4) = y1 + int(BUTTON_HVGAP*sin(pangle) + 0.5)
    polygon 4, sxv(), syv(), rgb(black), rgb(black)
  next i
end sub

' Play the current number of beeps and flash the corresponding buttons
sub PlayBeeps
  local i, note, button
  for i = 1 to nbeeps
    button = beeps(i)
    PlayBeep button, COMPUTER
  next i
end sub

' Play a single beep
sub PlayBeep button, who
  local note, flash, toval, foval
  flash = 1
  toval = option_values(OPT_TONLY, option_choice(OPT_TONLY))
  foval = option_values(OPT_FONLY, option_choice(OPT_FONLY))
  if toval then flash = 0
  note = button_tones(nbuttons-3, button)
  DrawButton button, flash
  if not foval then
    play tone note, note, beep_length
  end if
  pause beep_delay
  DrawButton button, 0
  if who = COMPUTER then pause beep_delay
end sub

' Handle multiple games
sub PlayLoop
  do
    if lclick then
      cb = GetControlButton()
      if cb > 0 then
        DoControlButton cb
      end if
      lclick = 0
    end if
    if start_play then
      start_play = 0
      InitGame 1
      PlayGame
    end if
  loop
end sub

' Initialize a new game
sub InitGame doF
  local i, c, b
print #1, "InitGame"
  c = option_choice(OPT_NBUTTONS)
  nbuttons = option_values(OPT_NBUTTONS, c)
  start_aflag = option_values(OPT_STARTA, option_choice(OPT_STARTA))
  MakeButtonPolygons nbuttons, start_aflag
  MakeButtonColors
  b = option_choice(OPT_NBEEPS)
  game_beeps = option_values(OPT_NBEEPS, option_choice(OPT_NBEEPS))
  select case option_choice(OPT_SPEED)
    case 1
      beep_length = 500
      beep_delay =  350
    case 2
      beep_length = 750
      beep_delay  = 500
    case 3
      beep_length = 350
      beep_delay = 250
  end select
  nbeeps = 0
  user_nbeeps = 0
  timeout = 0
  MakePrebeeps
  bccount = 0
  bcbutton = 0
  lclick = 0
  running = 1
  if doF then Flourish tlen, blen
end sub

' Play a game
sub PlayGame
  local tb
  local ptimer, utimer, allowed
  do
    if not running then exit do
    inc nbeeps
    beeps(nbeeps) = prebeeps(nbeeps)
    timer = 0
    pause SEQ_DELAY
    PlayBeeps
    ptimer = timer 
    user_nbeeps = 0
    allowed = SEQ_DELAY + ptimer + int(ptimer*UTIMEADD)
    timer = 0
    if USE_MOUSE then
      HandleUserClicks
    else
      HandleKeyboardEvents
    end if
    if running then
      utimer = timer
      if utimer > allowed then
        TooSlow
        exit do
      end if    
    end if
  loop until (nbeeps = game_beeps) or not running
end sub

' Make up a list of pseudo-random buttons
' This is a bit 'cooked' from a strictly uniform random
' distribution to get rid of excessive consecutive beeps
' on the same button.
sub MakePrebeeps
  local i, j, t, r
  for i = 1 to MAX_BEEPS
    beeps(i) = 0
  next i
  for i = 1 to game_beeps
    prebeeps(i) = ((i-1) mod nbuttons) + 1
  next i
  for i = 1 to NSHUFFLES
    for j = 1 to game_beeps-1
      r = RandInt(1, game_beeps)
      t = prebeeps(r)
      prebeeps(r) = prebeeps(j)
      prebeeps(j) = t
    next j
  next i
end sub

' Handle a user mouse button clicks until sequence complete
' or a wrong button is pressed.
sub HandleUserClicks
  local button, cb, bkw, bx
  bkw = option_values(OPT_BACKW, option_choice(OPT_BACKW))
  do
    if lclick then
      button = GetClickedButton()
      if button > 0 then
        inc user_nbeeps
        PlayBeep button, USER
        if bkw then
          bx = nbeeps - user_nbeeps + 1
        else
          bx = user_nbeeps
        end if
        if button <> beeps(bx) then
          running = 0
          ShowLostGame "Wrong button -- You lose!"
          exit sub
        else
        end if
        if user_nbeeps = game_beeps then
          ShowWonGame
          exit sub
        end if
      end if
      lclick = 0
    end if
  loop until (user_nbeeps = nbeeps) or not running
end sub

' analyze mouse click and return the button index,
' or zero if not in a button
function GetClickedButton()
  local float radius, angle, wangle, tangle, tinc, t1, t2, sangle
  local x, y, b
  sangle = 0
  if start_aflag then sangle = 0.5*(360.0/nbuttons)
  GetClickedButton = 0
  x = mouse_x - cx
  y = mouse_y - cy
  radius = sqr(x*x + y*y)
  if radius < BUTTON_INNER_RAD or radius > BUTTON_OUTER_RAD then
    exit function
  end if
  angle = GetAngle(x, y)
  tangle = sangle
  tinc = 360.0/nbuttons
  wangle = tinc
  for i = 1 to nbuttons
    if angle >= tangle and angle <= tangle+wangle then
      GetClickedButton = i
      exit function
    end if
    if tangle+wangle > 360.0 then
      t1 = 0.0 : t2 = tangle+wangle-360.0
      if angle >= t1 and angle <= t2 then
        GetClickedButton = i
        exit function
      end if
    end if
    inc tangle, tinc
  next i
end function

' handle keyboard events
sub HandleKeyboardEvents
  local z$, cmd, button
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    button = 0
    cmd = asc(UCASE$(z$))
    select case cmd
      case KQ
        button = 2
      case KW
        button = 1
      case KA
        button = 3
      case KS
        button = 4
      case KI
        button = 2
      case KO
        button = 1
      case KK
        button = 3
      case KL
        button = 4
    end select
    if button > 0 then
      inc user_nbeeps
      PlayBeep button, USER
      if bkw then
        bx = nbeeps - user_nbeeps + 1
      else
        bx = user_nbeeps
      end if
      if button <> beeps(bx) then
        running = 0
        ShowLostGame "Wrong button -- You lose!"
        exit sub
      end if
      if user_nbeeps = game_beeps then
        ShowWonGame
        exit sub
      end if
    end if
  loop until (user_nbeeps = nbeeps) or not running
end sub

' handle timeout on user input
sub TooSlow
  timeout = 1
  running = 0
  pause 100
  ShowLostGame "You Are Too Slow -- You Lose!"
end sub

' show the user losing
sub ShowLostGame m$
  running = 0
  ShowMessage m$, rgb(red), rgb(black)
  PlayTune LOSE_TUNE
  DrawScreen
end sub

' show the user winning
sub ShowWonGame
  pause 100
  running = 0
  ShowMessage "You Win!!", rgb(green), rgb(black)
  PlayTune WIN_TUNE
  DrawScreen
end sub

' Show a text message
sub ShowMessage m$, bc, tc
  local bw = 400
  local bh = 200
  box mm.hres\2 - bw\2, mm.vres\2 - bh\2, bw, bh,, rgb(black), bc
  text mm.hres\2, mm.vres\2, m$, "CM", 4,, tc, -1
end sub

' Play a tune for winning or losing
sub PlayTune which
  local i
  select case which
    case WIN_TUNE
      for i = 1 to NWNOTES
        play tone wnotes(i), wnotes(i), wlengths(i)
        pause wlengths(i) + 50
      next i
    case LOSE_TUNE
      for i = 1 to NDNOTES
        play tone dnotes(i), dnotes(i), dlengths(i)
        pause dlengths(i) + 50
      next i
  end select
end sub

' Start off with a flourish of lights and tones
sub Flourish blen, dlen
  local i, b, sbl, sdl
  sbl = beep_length : sdl = beep_delay
  beep_length = 200
  beep_delay = 100
  for i = 1 to nbuttons
    b = ((i-1) mod nbuttons) + 1
    PlayBeep b, COMPUTER
  next i
  beep_length = sbl : beep_delay = sdl
end sub

' Return the control button index or zero if not a click
' in a control button
function GetControlButton()
  bx = 0
  GetControlButton = bx
  if mouse_y < CTBY or mouse_y > CTBY+CTBH then exit function
  if mouse_x < HLPX or mouse_x > HLPX + CTBTW then exit function
  if mouse_x >= HLPX and mouse_x <= HLPX+CTBW then bx = 1
  if mouse_x >= OPTX and mouse_x <= OPTX+CTBW then bx = 2
  if mouse_x >= PLAX and mouse_x <= PLAX+CTBW then bx = 3
  if mouse_x >= QITX and mouse_x <= QITX+CTBW then bx = 4
  DrawControls bx
  pause 100
  DrawControls 0
  GetControlButton = bx
end function

' Handle Control Button Clicks
sub DoControlButton which
  running = 0
  select case which
    case 1 : ShowInstructions
    case 2 : ShowOptions
    case 3 : start_play = 1
    case 4 : Quit
  end select
end sub
        
' return a uniformly distributed random integer in the specified closed range
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

' Convert an HSV value to its RGB equivalent
' The S and V values must be in range 0..1; the H value must
' be in range 0..360. The RGB values will be in range 0..255.
sub HSV2RGB h as float, s as float, v as float, r, g, b
  local float i, hh, f, p, q, t, x, c, rp, gp, bp
  c = v*s
  hh = h/60.0
  i = int(hh)
  f = hh - i
  p = v*(1-s)
  q = v*(1-s*f)
  t = v*(1-s*(1-f))
  x = c*(1.0 - hh MOD 2 - 1)
  
  select case i
    case 0
      rp = v : gp = t : bp = p
    case 1
      rp = q : gp = v : bp = p
    case 2
      rp = p : gp = v : bp = t
    case 3
      rp = p : gp = q : bp = v
    case 4
      rp = t : gp = p : bp = v
    case 5
      rp = v : gp = p : bp = q
  end select
  r = rp*255.0 : g = gp*255.0 : b = bp*255.0
end sub

' Given X and Y offsets, compute the angle
' Handles the necessary issues with singularities of atan2()
' and flips needed for different quadrants.
' WARNING: assuming option angle degrees!
function GetAngle(dx, dy) as float
  local integer q
  local float a, c
  if dx = 0 and dy = 0 then
    GetAngle = 0.0
    exit function
  end if
  if dx = 0 then
    if dy > 0 then
      a = 270.0
    else
      a = 270.0
    end if
  else
    q = Quadrant(dx, dy)
    if dy = 0 then
      if dx > 0 then
        c = 0.0
      else
        c = 180.0
      end if
    else
      c = atan2(dy, dx)
    end if
    select case q
      case 1, 2
        a = -c
      case 3, 4
        a = 360.0 - c
    end select
  end if
  GetAngle = a
end function

' Given an x and y offset, return the corresponding quadrant.
' This is used to correct angle values returned from atan2().
function Quadrant(dx, dy)
  local q = 0
    if dy < 0 and dx >= 0 then q = 1
  if dy < 0 and dx < 0 then q = 2
  if dy >= 0 and dx < 0 then q = 3
  if dy > 0 and dx > 0 then q = 4
  Quadrant = q
end function

' Show Instructions
sub ShowInstructions
  local z$
  cls
  text mm.hres\2, 10, "Instructions for Playing Simon", "CT", 4,, rgb(green)
  print @(0, 40) "NOMIS is a near-clone of the classic Simon game."
  print ""
  print "The 4 or more large colored buttons play tones when pressed, either by you or by the"
  print "computer. A game starts with the computer playing a tone and flashing the"
  print "corresponding button. You must press that same button to keep playing. Next,
  print "the computer plays and flashes two buttons, starting with the same one as before."
  print "You have to press those same 2 buttons. As the game progresses, the sequence of"
  print "button presses and tones gets longer and longer. If you press the wrong button or"
  print "hesitate too long, the game is over.
  print ""
  print "The normal game stops after 8 tones. If you get them all correctly, you win!"
  print ""
  print "There are lots of options you can choose to make the game more challenging and more"
  print "interesting:"
  print ""
  print "  Number of Buttons: (4 to 8)          [ Default = 4 ]
  print "  Number of Tones: (8, 14, 20)         [ Default = 8 ]
  print "  Speed: (normal, slow, fast)          [ Default = normal]
  print "  Suppress tones: (0=No, 1=Yes)        [ Default = No]
  print "  Suppress flashes: (0=No, 1=Yes)      [ Default = No]
  print "  Play tones BACKWARDS: (0=No, 1=Yes)  [ Default = No]
  print "  Start Angle: (0=normal, 1=rotate)    [ Default = No]
  print ""
  print "(and yes, you can suppress BOTH the tones and the button flashes. I guess this counts"
  print "as the most difficult game of all :-)"
  print ""
  print "If the Start Angle is set to 'rotate', the buttons are rotated by one-half of their"
  print "angular span; e.g. 45 degrees for 4 buttons, 30 degrees for 6 buttons, etc.
  print ""
  print "You can play NOMIS using a mouse or with the keyboard alone. If you don't have a mouse,"
  print "open the 'NOMIS.bas' source file and set the constant USE_MOUSE to zero at the top"
  print "of the source file. When playing using a keyboard only, you are restricted to just"
  print "four buttons as opposed to using the mouse, where you can configure from 4 to 8 buttons."
  print ""
  print "Two sets of keys on the keyboard stand for the 4 NOMIS colored buttons: you can use"
  print "either the Q, W, A, and S keys or the I, O, K, and L keys, or any combination. These"
  print "sets of 4 keys are chosen so that their physical arrangement on the keyboard is similar"
  print "to the NOMIS buttons. So for instance, Q stands for the top left button, W for the top"
  print "right button, A for the bottom left, and S for the bottom right.
  print ""
  print "If you are using a mouse, 4 control buttons are displayed at the top of the screen, for"
  print "Help, Options, Play, and Quit. If you are using a keyboard only, the corresponding"
  print "keyboard keys are F1, F2, F4, and Escape.
  text mm.hres\2, 585, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  DrawScreen
end sub

' Show and let user choose options
sub ShowOptions
  local z$, cmd, sel
  cls
  text mm.hres\2, 10, "Game Play Options", "CT", 4,, rgb(green)
  print @(0, 40) "Press the Left or Right Arrow key to cycle through the values"
  print "for an option category. Press the Up or Down arrow key to move to different"
  print "categories. Press Enter to accept the current options and return to the game."
  sel = 1
  DrawOptions sel
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        if sel > 1 then
          inc sel, -1
        else
          sel = NOPTS
        end if
        DrawOptions sel
      case DOWN
        if sel < NOPTS then
          inc sel
        else
          sel = 1
        end if
        DrawOptions sel
      case LEFT
        if option_choice(sel) > 1 then
          inc option_choice(sel), -1
        else
          option_choice(sel) = noptvals(sel)
        end if
        DrawOptions sel
      case RIGHT
        if option_choice(sel) < noptvals(sel)  then
          inc option_choice(sel), 1
        else
          option_choice(sel) = 1
        end if
        DrawOptions sel
      case ENTER
        exit do
    end select
  loop
  InitGame 0
  DrawScreen
end sub      

' Draw the Options
sub DrawOptions sel
  local i, tc, bc, m$, v, c
  tc = rgb(white) : bc = rgb(black)
  for i = 1 to NOPTS
    c = option_choice(i)
    v = option_values(i, c)
    m$ = option_text$(i) + " " + str$(v) + " "
    if i = sel then
      text 30, 100+(i-1)*30, m$,,,, bc, tc
    else
      text 30, 100+(i-1)*30, m$,,,, tc, bc
    end if
  next i
end sub

' Quit cleanly
sub Quit
  gui cursor off
  controller mouse close
  cls
  end
end sub

' Note frequencies
' white keys
data 87.307, 98.0, 110.0, 123.74, 130.81, 146.83, 164.81, 174.61, 196.0, 220.0, 246.94, 261.63
data 293.66, 329.63, 349.23, 392.0, 440.0, 493.88, 523.25, 587.33, 659.26, 698.456
' black keys
data 92.499, 103.826, 116.541, 138.591, 155.563, 184.997, 207.652, 233.082, 277.183, 311.127, 369.994
data 415.305, 466.164, 554.183, 622.254, 739.989

' Scales for different numbers of buttons (1= white key, 0 = black key)
data 1, 14, 0, 9, 1, 10, 1, 7                         ' standard 4 button scale
data 1, 12, 1, 13, 1, 14, 1, 16, 1, 17                ' major pentatonic
data 1, 12, 1, 13, 1, 14, 0, 11, 0, 12, 0, 13         ' hexatonic
data 1, 10, 1, 11, 1, 12, 1, 13, 1, 14, 1, 15, 1, 16  ' A minor
data 1, 12, 1, 13, 1, 14, 1, 15, 1, 16, 1, 17, 1, 18, 1, 19  ' C major diatonic
